<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Quotation;
use App\Models\QuotationItem;
use App\Models\Setting;
use App\Models\User;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;

class QuotationController extends Controller
{
    public function index(Request $request)
    {
        $query = Quotation::query();

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('quotation_number', 'like', "%{$search}%")
                    ->orWhere('customer_name', 'like', "%{$search}%")
                    ->orWhere('customer_email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $quotations = $query->latest()->paginate(15);

        return view('admin.quotations.index', compact('quotations'));
    }

    public function create(Request $request)
    {
        $customers = User::customers()->orderBy('name')->get();
        $products = Product::active()->orderBy('name')->get();
        $defaultDiscount = Setting::getValue('installation_discount', 10);
        $type = $request->get('type', 'products'); // 'products' or 'services'

        return view('admin.quotations.create', compact('customers', 'products', 'defaultDiscount', 'type'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email',
            'customer_phone' => 'required|string|max:20',
            'customer_address' => 'nullable|string',
            'site_address' => 'nullable|string',
            'service_type' => 'nullable|string',
            'service_description' => 'nullable|string',
            'valid_until' => 'nullable|date',
            'notes' => 'nullable|string',
            'is_shop_customer' => 'boolean',
            'discount_percentage' => 'required|numeric|min:0|max:100',
            'items' => 'required|array|min:1',
            'items.*.item_type' => 'required|in:product,service,labor',
            'items.*.description' => 'required|string',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        $quotation = Quotation::create([
            'quotation_number' => Quotation::generateQuotationNumber(),
            'user_id' => $request->user_id,
            'status' => 'draft',
            'customer_name' => $validated['customer_name'],
            'customer_email' => $validated['customer_email'],
            'customer_phone' => $validated['customer_phone'],
            'customer_address' => $validated['customer_address'] ?? null,
            'site_address' => $validated['site_address'] ?? null,
            'service_type' => $validated['service_type'] ?? null,
            'service_description' => $validated['service_description'] ?? null,
            'valid_until' => $validated['valid_until'] ?? now()->addDays(30),
            'notes' => $validated['notes'] ?? null,
            'is_shop_customer' => $validated['is_shop_customer'] ?? false,
            'discount_percentage' => $validated['discount_percentage'],
            'terms_conditions' => Setting::getValue('default_terms'),
            'products_total' => 0,
            'installation_total' => 0,
            'installation_discount' => 0,
            'total' => 0,
        ]);

        foreach ($validated['items'] as $item) {
            QuotationItem::create([
                'quotation_id' => $quotation->id,
                'product_id' => $item['product_id'] ?? null,
                'item_type' => $item['item_type'],
                'description' => $item['description'],
                'unit_price' => $item['unit_price'],
                'quantity' => $item['quantity'],
                'subtotal' => $item['unit_price'] * $item['quantity'],
            ]);
        }

        $quotation->calculateTotals();

        return redirect()->route('admin.quotations.show', $quotation)
            ->with('success', 'Quotation created successfully.');
    }

    public function show(Quotation $quotation)
    {
        $quotation->load('items.product', 'user');
        return view('admin.quotations.show', compact('quotation'));
    }

    public function edit(Quotation $quotation)
    {
        $quotation->load('items');
        $customers = User::customers()->orderBy('name')->get();
        $products = Product::active()->orderBy('name')->get();

        return view('admin.quotations.edit', compact('quotation', 'customers', 'products'));
    }

    public function update(Request $request, Quotation $quotation)
    {
        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email',
            'customer_phone' => 'required|string|max:20',
            'customer_address' => 'nullable|string',
            'site_address' => 'nullable|string',
            'service_type' => 'nullable|string',
            'service_description' => 'nullable|string',
            'valid_until' => 'nullable|date',
            'notes' => 'nullable|string',
            'is_shop_customer' => 'boolean',
            'discount_percentage' => 'required|numeric|min:0|max:100',
            'status' => 'required|in:draft,sent,accepted,rejected,expired',
            'items' => 'required|array|min:1',
            'items.*.item_type' => 'required|in:product,service,labor',
            'items.*.description' => 'required|string',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        $quotation->update([
            'customer_name' => $validated['customer_name'],
            'customer_email' => $validated['customer_email'],
            'customer_phone' => $validated['customer_phone'],
            'customer_address' => $validated['customer_address'] ?? null,
            'site_address' => $validated['site_address'] ?? null,
            'service_type' => $validated['service_type'] ?? null,
            'service_description' => $validated['service_description'] ?? null,
            'valid_until' => $validated['valid_until'] ?? null,
            'notes' => $validated['notes'] ?? null,
            'is_shop_customer' => $validated['is_shop_customer'] ?? false,
            'discount_percentage' => $validated['discount_percentage'],
            'status' => $validated['status'],
        ]);

        // Sync items
        $quotation->items()->delete();

        foreach ($validated['items'] as $item) {
            QuotationItem::create([
                'quotation_id' => $quotation->id,
                'product_id' => $item['product_id'] ?? null,
                'item_type' => $item['item_type'],
                'description' => $item['description'],
                'unit_price' => $item['unit_price'],
                'quantity' => $item['quantity'],
                'subtotal' => $item['unit_price'] * $item['quantity'],
            ]);
        }

        $quotation->calculateTotals();

        return redirect()->route('admin.quotations.show', $quotation)
            ->with('success', 'Quotation updated successfully.');
    }

    public function destroy(Quotation $quotation)
    {
        $quotation->delete();

        return redirect()->route('admin.quotations.index')
            ->with('success', 'Quotation deleted successfully.');
    }

    public function downloadPdf(Quotation $quotation)
    {
        $quotation->load('items.product');
        $company = Setting::getByGroup('company');
        $payment = Setting::getByGroup('payment');

        $pdf = Pdf::loadView('admin.quotations.pdf', compact('quotation', 'company', 'payment'));

        return $pdf->download("quotation-{$quotation->quotation_number}.pdf");
    }

    public function sendEmail(Quotation $quotation)
    {
        // TODO: Implement email sending
        $quotation->update(['status' => 'sent']);

        return back()->with('success', 'Quotation marked as sent.');
    }
}
