<?php

namespace App\Http\Controllers\Shop;

use App\Http\Controllers\Controller;
use App\Mail\OrderInvoice;
use App\Models\CartItem;
use App\Models\Order;
use App\Models\OrderItem;
use App\Services\InvoiceService;
use App\Services\WhatsAppService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class CheckoutController extends Controller
{
    public function __construct(
        protected InvoiceService $invoiceService,
        protected WhatsAppService $whatsAppService
    ) {}

    /**
     * Process the checkout.
     */
    public function store(Request $request)
    {
        $user = auth()->user();
        $cartItems = CartItem::with('product')->where('user_id', $user->id)->get();

        if ($cartItems->isEmpty()) {
            return redirect()->route('cart.index')->with('error', 'Your cart is empty.');
        }

        try {
            DB::beginTransaction();

            // Calculate totals
            $subtotal = $cartItems->sum('subtotal');
            $installationTotal = $cartItems->sum('installation_cost');
            $total = $subtotal + $installationTotal;

            // Create Order
            $order = Order::create([
                'order_number' => Order::generateOrderNumber(),
                'user_id' => $user->id,
                'status' => 'pending',
                'payment_status' => 'pending',
                'customer_name' => $user->name,
                'customer_email' => $user->email,
                'customer_phone' => $user->phone,
                'shipping_address' => $user->address,
                'city' => $user->city,
                'county' => $user->county,
                'subtotal' => $subtotal,
                'installation_cost' => $installationTotal,
                'total' => $total,
            ]);

            foreach ($cartItems as $item) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item->product_id,
                    'product_name' => $item->product->name,
                    'product_sku' => $item->product->sku,
                    'price' => $item->product->effective_price,
                    'quantity' => $item->quantity,
                    'subtotal' => $item->product->effective_price * $item->quantity, // subtotal is just price * qty
                    'includes_installation' => $item->include_installation,
                    'installation_price' => $item->product->installation_price ?? 0,
                ]);

                // Deduct stock
                $item->product->decrement('stock_quantity', $item->quantity);
            }

            // Clear Cart
            CartItem::where('user_id', $user->id)->delete();

            DB::commit();

            // Send Notifications (Queue these in production)
            
            // 1. Email to Customer & Company
            // Send to customer
            try {
                Mail::to($user->email)->send(new OrderInvoice($order, $this->invoiceService));
            } catch (\Exception $e) {
                Log::error('Failed to send invoice email to customer (' . $user->email . '): ' . $e->getMessage());
            }

            // Send to company invoice email
            try {
                Mail::to('info@onespotsolutionsltd.co.ke')->send(new OrderInvoice($order, $this->invoiceService));
            } catch (\Exception $e) {
                Log::error('Failed to send invoice email to admin (info@onespotsolutionsltd.co.ke): ' . $e->getMessage());
            }

            // Generate WhatsApp link for order notification
            $whatsappMessage = $this->whatsAppService->generateAdminOrderMessage($order);
            $whatsappLink = $this->whatsAppService->generateAdminLink($whatsappMessage);

            // Redirect to confirmation page with WhatsApp link
            return redirect()->route('shop.order.confirmation', ['order' => $order->order_number])
                ->with('whatsapp_link', $whatsappLink);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Checkout failed: ' . $e->getMessage());
            return back()->with('error', 'Something went wrong while processing your order. Please try again.');
        }
    }

    /**
     * Show order confirmation page with WhatsApp link.
     */
    public function confirmation(string $order)
    {
        $orderModel = Order::with(['items', 'user'])->where('order_number', $order)->firstOrFail();
        
        // Regenerate WhatsApp link for admin notification if not in session
        $whatsappLink = session('whatsapp_link');
        if (!$whatsappLink) {
            $whatsappMessage = $this->whatsAppService->generateAdminOrderMessage($orderModel);
            $whatsappLink = $this->whatsAppService->generateAdminLink($whatsappMessage);
        }

        // Generate WhatsApp link to send invoice to customer using their registered phone
        $customerWhatsappLink = $this->whatsAppService->generateCustomerLinkFromOrder($orderModel);

        return view('shop.order-confirmation', [
            'order' => $orderModel,
            'whatsappLink' => $whatsappLink,
            'customerWhatsappLink' => $customerWhatsappLink,
        ]);
    }
}
