<?php

namespace App\Http\Controllers\Shop;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;

class ShopController extends Controller
{
    /**
     * Display the shop homepage.
     */
    public function home()
    {
        $featuredProducts = Product::with(['category', 'images'])
            ->where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->take(8)
            ->get();

        $categories = Category::withCount('products')
            ->whereNull('parent_id')
            ->get();

        return view('shop.home', compact('featuredProducts', 'categories'));
    }

    /**
     * Display all products with optional filtering.
     */
    public function products(Request $request)
    {
        $query = Product::with(['category', 'images'])
            ->where('is_active', true);

        // Filter by category
        if ($request->has('category') && $request->category) {
            $query->where('category_id', $request->category);
        }

        // Sort options
        $sortBy = $request->get('sort', 'newest');
        switch ($sortBy) {
            case 'price_low':
                $query->orderBy('price', 'asc');
                break;
            case 'price_high':
                $query->orderBy('price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            default:
                $query->orderBy('created_at', 'desc');
        }

        $products = $query->paginate(12);
        $categories = Category::whereNull('parent_id')->get();

        return view('shop.products', compact('products', 'categories'));
    }

    /**
     * Display all categories.
     */
    public function categories()
    {
        $categories = Category::with(['children', 'products'])
            ->whereNull('parent_id')
            ->withCount('products')
            ->get();

        return view('shop.categories', compact('categories'));
    }

    /**
     * Display the installation/services page.
     */
    public function installation()
    {
        return view('shop.installation');
    }

    /**
     * Display the contact page.
     */
    public function contact()
    {
        return view('shop.contact');
    }

    /**
     * Handle contact form submission.
     */
    public function sendContactMessage(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:2000',
        ]);

        try {
            \Illuminate\Support\Facades\Mail::send([], [], function ($mail) use ($validated) {
                $mail->to('info@onespotsolutionsltd.co.ke')
                    ->replyTo($validated['email'], $validated['name'])
                    ->subject('Contact Form: ' . $validated['subject'])
                    ->html("
                        <h2>New Contact Form Submission</h2>
                        <p><strong>From:</strong> {$validated['name']}</p>
                        <p><strong>Email:</strong> {$validated['email']}</p>
                        <p><strong>Phone:</strong> " . ($validated['phone'] ?? 'Not provided') . "</p>
                        <p><strong>Subject:</strong> {$validated['subject']}</p>
                        <hr>
                        <p><strong>Message:</strong></p>
                        <p>" . nl2br(e($validated['message'])) . "</p>
                    ");
            });
            
            return back()->with('success', 'Thank you for your message! We will get back to you soon.');
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Contact form email failed: ' . $e->getMessage());
            return back()->with('success', 'Thank you for your message! We will get back to you soon.');
        }
    }

    /**
     * Display a single product.
     */
    public function showProduct(Product $product)
    {
        if (!$product->is_active) {
            abort(404);
        }

        $product->load(['category', 'images']);
        
        $relatedProducts = Product::with(['category', 'images'])
            ->where('is_active', true)
            ->where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->take(4)
            ->get();

        return view('shop.product', compact('product', 'relatedProducts'));
    }
}
