<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Payment extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'payment_method',
        'transaction_id',
        'merchant_request_id',
        'checkout_request_id',
        'amount',
        'phone_number',
        'bank_reference',
        'status',
        'response_data',
        'failure_reason',
        'paid_at',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'response_data' => 'array',
        'paid_at' => 'datetime',
    ];

    /**
     * Order relationship
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Check if payment is successful
     */
    public function getIsSuccessfulAttribute(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Check if payment is M-Pesa
     */
    public function getIsMpesaAttribute(): bool
    {
        return $this->payment_method === 'mpesa';
    }

    /**
     * Check if payment is bank transfer
     */
    public function getIsBankAttribute(): bool
    {
        return $this->payment_method === 'bank';
    }

    /**
     * Mark payment as completed
     */
    public function markAsCompleted(string $transactionId = null): void
    {
        $this->update([
            'status' => 'completed',
            'transaction_id' => $transactionId ?? $this->transaction_id,
            'paid_at' => now(),
        ]);

        // Update order payment status
        $this->order->update(['payment_status' => 'paid']);
    }

    /**
     * Mark payment as failed
     */
    public function markAsFailed(string $reason = null): void
    {
        $this->update([
            'status' => 'failed',
            'failure_reason' => $reason,
        ]);

        $this->order->update(['payment_status' => 'failed']);
    }

    /**
     * Status badge
     */
    public function getStatusBadgeAttribute(): array
    {
        $badges = [
            'pending' => ['color' => 'warning', 'label' => 'Pending'],
            'completed' => ['color' => 'success', 'label' => 'Completed'],
            'failed' => ['color' => 'danger', 'label' => 'Failed'],
            'cancelled' => ['color' => 'secondary', 'label' => 'Cancelled'],
        ];

        return $badges[$this->status] ?? ['color' => 'secondary', 'label' => $this->status];
    }
}
