<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Quotation extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'quotation_number',
        'user_id',
        'status',
        'customer_name',
        'customer_email',
        'customer_phone',
        'customer_address',
        'service_type',
        'service_description',
        'site_address',
        'products_total',
        'installation_total',
        'installation_discount',
        'discount_percentage',
        'total',
        'valid_until',
        'terms_conditions',
        'notes',
        'is_shop_customer',
    ];

    protected $casts = [
        'products_total' => 'decimal:2',
        'installation_total' => 'decimal:2',
        'installation_discount' => 'decimal:2',
        'discount_percentage' => 'decimal:2',
        'total' => 'decimal:2',
        'valid_until' => 'date',
        'is_shop_customer' => 'boolean',
    ];

    /**
     * User who requested quotation
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Quotation items
     */
    public function items(): HasMany
    {
        return $this->hasMany(QuotationItem::class);
    }

    /**
     * Generate unique quotation number
     */
    public static function generateQuotationNumber(): string
    {
        $prefix = 'QUO';
        $date = now()->format('Ymd');
        $count = self::whereDate('created_at', today())->count() + 1;
        return sprintf('%s-%s-%04d', $prefix, $date, $count);
    }

    /**
     * Calculate totals with shop customer discount
     */
    public function calculateTotals(): void
    {
        $productsTotal = $this->items()->where('item_type', 'product')->sum('subtotal');
        $servicesTotal = $this->items()->whereIn('item_type', ['service', 'labor'])->sum('subtotal');
        
        $discountAmount = 0;
        if ($this->is_shop_customer && $servicesTotal > 0) {
            $discountAmount = $servicesTotal * ($this->discount_percentage / 100);
        }

        $this->update([
            'products_total' => $productsTotal,
            'installation_total' => $servicesTotal,
            'installation_discount' => $discountAmount,
            'total' => $productsTotal + $servicesTotal - $discountAmount,
        ]);
    }

    /**
     * Check if quotation is expired
     */
    public function getIsExpiredAttribute(): bool
    {
        return $this->valid_until && $this->valid_until->isPast();
    }

    /**
     * Status badge
     */
    public function getStatusBadgeAttribute(): array
    {
        $badges = [
            'draft' => ['color' => 'secondary', 'label' => 'Draft'],
            'sent' => ['color' => 'primary', 'label' => 'Sent'],
            'accepted' => ['color' => 'success', 'label' => 'Accepted'],
            'rejected' => ['color' => 'danger', 'label' => 'Rejected'],
            'expired' => ['color' => 'warning', 'label' => 'Expired'],
        ];

        return $badges[$this->status] ?? ['color' => 'secondary', 'label' => $this->status];
    }

    /**
     * Scope for active quotations
     */
    public function scopeActive($query)
    {
        return $query->whereIn('status', ['draft', 'sent']);
    }
}
