<?php

namespace App\Services;

use App\Models\Order;
use App\Models\Product;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class OrderProcessingService
{
    /**
     * Parse message content to extract order number and action
     * Expected formats:
     * - "PROCESSED OSS-20260116-XXXX"
     * - "REJECTED OSS-20260116-XXXX"
     */
    public function parseMessage(string $content): ?array
    {
        // Normalize content - uppercase and trim
        $content = strtoupper(trim($content));
        
        // Pattern to match action and order number
        // Matches: PROCESSED/REJECTED followed by order number (OSS-YYYYMMDD-XXXX)
        $pattern = '/\b(PROCESSED|REJECTED)\s+(OSS-\d{8}-[A-Z0-9]{4})\b/i';
        
        if (preg_match($pattern, $content, $matches)) {
            return [
                'action' => strtolower($matches[1]),
                'order_number' => $matches[2],
            ];
        }
        
        return null;
    }

    /**
     * Find order by order number
     */
    public function findOrder(string $orderNumber): ?Order
    {
        return Order::where('order_number', $orderNumber)->first();
    }

    /**
     * Process an order - update status to 'processing' and deduct stock
     */
    public function processOrder(Order $order, string $channel = 'email'): array
    {
        // Check if order is in pending status
        if ($order->status !== 'pending') {
            return [
                'success' => false,
                'message' => "Order {$order->order_number} is not in pending status. Current status: {$order->status}",
            ];
        }

        try {
            DB::beginTransaction();

            // Load order items with products
            $order->load('items.product');

            // Check stock availability for all items
            foreach ($order->items as $item) {
                if ($item->product && $item->product->stock_quantity < $item->quantity) {
                    DB::rollBack();
                    return [
                        'success' => false,
                        'message' => "Insufficient stock for product: {$item->product_name}. Available: {$item->product->stock_quantity}, Required: {$item->quantity}",
                    ];
                }
            }

            // Deduct stock for each item
            foreach ($order->items as $item) {
                if ($item->product) {
                    $item->product->decrement('stock_quantity', $item->quantity);
                    
                    // Update stock status if needed
                    if ($item->product->stock_quantity <= 0) {
                        $item->product->update(['stock_status' => 'out_of_stock']);
                    }
                }
            }

            // Update order status
            $order->update([
                'status' => 'processing',
                'processed_at' => now(),
                'processed_via' => $channel,
            ]);

            DB::commit();

            // Log the action
            $this->logAction('processed', $order, $channel);

            return [
                'success' => true,
                'message' => "Order {$order->order_number} has been processed successfully. Stock updated.",
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Order processing failed', [
                'order_number' => $order->order_number,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => "Failed to process order: {$e->getMessage()}",
            ];
        }
    }

    /**
     * Reject an order - update status to 'cancelled' without affecting stock
     */
    public function rejectOrder(Order $order, string $channel = 'email'): array
    {
        // Check if order is in pending status
        if ($order->status !== 'pending') {
            return [
                'success' => false,
                'message' => "Order {$order->order_number} is not in pending status. Current status: {$order->status}",
            ];
        }

        try {
            // Update order status to cancelled
            $order->update([
                'status' => 'cancelled',
                'processed_at' => now(),
                'processed_via' => $channel,
            ]);

            // Log the action
            $this->logAction('rejected', $order, $channel);

            return [
                'success' => true,
                'message' => "Order {$order->order_number} has been cancelled. Stock unchanged.",
            ];

        } catch (\Exception $e) {
            Log::error('Order rejection failed', [
                'order_number' => $order->order_number,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => "Failed to reject order: {$e->getMessage()}",
            ];
        }
    }

    /**
     * Handle incoming message from any channel
     */
    public function handleMessage(string $content, string $channel): array
    {
        // Parse the message
        $parsed = $this->parseMessage($content);

        if (!$parsed) {
            return [
                'success' => false,
                'message' => 'Invalid message format. Expected: PROCESSED OSS-XXXXXX-XXXX or REJECTED OSS-XXXXXX-XXXX',
            ];
        }

        // Find the order
        $order = $this->findOrder($parsed['order_number']);

        if (!$order) {
            return [
                'success' => false,
                'message' => "Order not found: {$parsed['order_number']}",
            ];
        }

        // Process based on action
        if ($parsed['action'] === 'processed') {
            return $this->processOrder($order, $channel);
        } else {
            return $this->rejectOrder($order, $channel);
        }
    }

    /**
     * Log processing action for audit trail
     */
    protected function logAction(string $action, Order $order, string $channel): void
    {
        Log::info('Order processing action', [
            'action' => $action,
            'order_number' => $order->order_number,
            'order_id' => $order->id,
            'channel' => $channel,
            'processed_at' => now()->toDateTimeString(),
            'customer_name' => $order->customer_name,
            'customer_email' => $order->customer_email,
            'total' => $order->total,
        ]);
    }
}
